--
-- This file contains both Oracle and SQL Server sql statements that
-- illustrate the differences between PL/SQL and Transact-SQL and how
-- you can migrate database code from one database type to the other. 
--

--
-- Creating an Oracle table based upon a SELECT statement
--


CREATE TABLE DALLAS_EMPLOYEES 
select a.empno, a.ename, a.job, a.mgr, a.hiredate,
       a.sal, a.comm, a.deptno 
from emp a, dept b
where a.deptno = b.deptno and
      b.loc = 'DALLAS'
;

--
-- Creating a SQL Server table based upon a SELECT INTO statement
--

select EmployeeID, LastName, FirstName, Title, Country, ReportsTo
into LONDON_EMPLOYEES
from Employees
where City = 'LONDON'
go


--
-- Oracle statement to traverse a hierarchical relationship using the
-- CONNECT BY clause
--

select hierarchy_name
from hierarchy
start with child='Root'
connect by
prior child = parent
;


--
-- SQL Server procedure to traverse a hierarchical relationship 
-- 

CREATE PROCEDURE expand (@current char(20)) 
as
SET NOCOUNT ON
DECLARE 
   @level int, 
   @line char(20)

CREATE TABLE #stack (item char(20), level int)

INSERT INTO #stack VALUES (@current, 1)

SELECT @level = 1
WHILE @level > 0
BEGIN
  IF EXISTS (SELECT * FROM #stack WHERE level = @level)
  BEGIN
    SELECT @current = item
    FROM #stack
    WHERE level = @level

    SELECT @line = space(@level - 1) + 
                  @current
    PRINT @line

    DELETE FROM #stack
    WHERE level = @level AND
          item = @current

    INSERT #stack
      SELECT child, @level + 1
      FROM hierarchy
      WHERE parent = @current

    IF @@ROWCOUNT > 0
       SELECT @level = @level + 1
  END
  ELSE
    SELECT @level = @level - 1
END -- WHILE
go

exec Expand 'Root'



--
-- Differences in Oracle and SQL Server syntax when writing a nested UPDATE statement
--

update table_one a
set last_updated_by = ( select user_id from my_users_table b
where a.author = b.user_name)
;

update a
set last_updated_by = b.user_id
from table_one a, my_user_table b
where a.author = b.user_name
go


--
-- Differences in Oracle and SQL Server syntax when performing string concatentation
--

select 'select count(*) from ' || table_name || ';' 
from user_tables
;

select 'select count(*) from ' + table_name 
from INFORMATION_SCHEMA.TABLES
go



--
-- Pattern matching differences in Oracle and SQL Server
--

select first_name, last_name from my_users_table
where first_name like ALL_N and
      last_name like SMITH%
;


select first_name, last_name from my_users_table
where first_name like [AE]LL[AE]N
go


--
-- OUTER JOIN syntax differences between Oracle and SQL Server
--


SELECT ename, job, emp.deptno, dname 
FROM emp, dept 
WHERE emp.deptno = dept.deptno(+)
;


-- 
-- In SQL Server the OUTER JOIN can be written using either of the following syntax  
-- 

SELECT ename, job, emp.deptno, dname 
FROM emp, dept 
WHERE emp.deptno *= dept.deptno
go

SELECT ename, job, emp.deptno, dname 
FROM emp 
RIGHT OUTER JOIN dept on emp.deptno = dept.deptno
go


--
-- Sample EMPLOYEES tables used to illustrate differences between PL/SQL and Transact SQL
--

CREATE TABLE EMPLOYEES ( 
 EMPLOYEE_ID     NUMBER(4),
 LAST_NAME       VARCHAR2(15),
 FIRST_NAME      VARCHAR2(15),
 JOB_ID          NUMBER(3),
 HIRE_DATE       DATE,
 SALARY          NUMBER(7,2)
)
;

CREATE TABLE EMPLOYEES (
 EMPLOYEE_ID     INT,
 LAST_NAME       VARCHAR(15),
 FIRST_NAME      VARCHAR(15),
 JOB_ID          SMALLINT,
 HIRE_DATE       DATETIME,
 SALARY          DECIMAL(7,2)
)
go


--
-- PL/SQL example
--

DECLARE 
  empID INTEGER;
  firstName varchar2(10);
  lastName  varchar2(20);
  jobTitle  varchar2(30);
  jobID     integer;
  hireDate  date;
  salary    number(7,2);
  newSalary    number(7,2);

  CURSOR EMPLOYEE_CURSOR IS
  SELECT EMPLOYEE_ID, FIRST_NAME, LAST_NAME, JOB_ID, HIRE_DATE, SALARY 
  FROM EMPLOYEES
  FOR UPDATE;

BEGIN

empID := 12345;

  BEGIN
     SELECT FIRST_NAME, LAST_NAME, SALARY
     INTO firstName, lastName, salary
     FROM EMPLOYEES 
     WHERE EMPLOYEE_ID = empID;

  EXCEPTION
     WHEN NO_DATA_FOUND
     THEN
        DBMS_OUTPUT.PUT_LINE( 'Employee ' || to_char(empID) ||
                              ' not found.');
  END;

-- MANAGERS working at least 8 years get a 25% raise
-- PRESIDENT gets a 100% raise
-- All others get 5% raise

DBMS_OUTPUT.PUT_LINE( 'Next');

OPEN EMPLOYEE_CURSOR;

FETCH EMPLOYEE_CURSOR
  INTO empID, firstName, lastName, jobID, hireDate, salary;

WHILE(EMPLOYEE_CURSOR%FOUND) LOOP

  IF( jobID = 667 ) THEN
     jobTitle := 'CLERK';
  ELSIF( jobID = 668 ) THEN
     jobTitle := 'STAFF';
  ELSIF( jobID = 669 ) THEN
     jobTitle := 'ANALYST';
  ELSIF( jobID = 670 ) THEN
     jobTitle := 'SALESPERSON';
  ELSIF( jobID = 671 ) THEN
     jobTitle := 'MANAGER';
  ELSIF( jobID = 672 ) THEN
     jobTitle := 'PRESIDENT';
  ELSE
     jobTitle := 'Unknown';
  END IF;

  DBMS_OUTPUT.PUT_LINE( 'Employee ' || firstName || ' ' || 
          lastName || ' holds the position of ' || jobTitle );

  IF( jobID = 672 ) THEN
     newSalary := salary * 2;
  ELSIF(jobID = 671 AND MONTHS_BETWEEN(sysdate,hireDate)>=96) THEN
     newSalary := salary * 1.25;
  ELSE
     newSalary := salary * 1.05;
  END IF;

  UPDATE EMPLOYEES
  SET SALARY = newSalary
  WHERE CURRENT OF EMPLOYEE_CURSOR;

  FETCH EMPLOYEE_CURSOR
    INTO empID, firstName, lastName, jobID, hireDate, salary;

END LOOP;

CLOSE EMPLOYEE_CURSOR;

EXCEPTION
   WHEN NO_DATA_FOUND
   THEN
      DBMS_OUTPUT.PUT_LINE( 'No data found');

   WHEN VALUE_ERROR
   THEN
      DBMS_OUTPUT.PUT_LINE( 'Invalid value specified.');

   WHEN OTHERS
   THEN
      DBMS_OUTPUT.PUT_LINE( 'An unknown error was encountered.');

END;
/

-- 
-- Transact-SQL example
--

DECLARE 
  @empID INT,
  @firstName varchar(10),
  @lastName  varchar(20),
  @jobTitle  varchar(30),
  @jobID     int,
  @hireDate  datetime,
  @newsalary decimal(7,2),
  @salary    decimal(7,2),
  @errno     int,
  @errmsg    varchar(255)


SET @empID = 12345

SELECT @firstName = FIRST_NAME,
       @lastName = LAST_NAME,
       @salary=SALARY
FROM EMPLOYEES 
WHERE EMPLOYEE_ID = @empID

DECLARE
  EMPLOYEE_CURSOR CURSOR FOR
  SELECT EMPLOYEE_ID, FIRST_NAME, LAST_NAME, JOB_ID, 
         HIRE_DATE, SALARY 
  FROM EMPLOYEES

-- MANAGERS working at least 8 years get a 25% raise
-- PRESIDENTS get a 100% raise
-- All others get 5% raise

OPEN EMPLOYEE_CURSOR

FETCH NEXT FROM EMPLOYEE_CURSOR
  INTO @empID, @firstName, @lastName, @jobID, @hireDate, @salary

WHILE(@@FETCH_STATUS <> -1)
BEGIN

  SELECT @jobTitle =
    CASE @jobID
      WHEN 667 THEN 'CLERK'
      WHEN 668 THEN 'STAFF'
      WHEN 669 THEN 'ANALYST'
      WHEN 670 THEN 'SALESPERSON'
      WHEN 671 THEN 'MANAGER'
      WHEN 672 THEN 'PRESIDENT'
      ELSE 'Unknown'
    END

  PRINT 'Employee ' + @firstName + ' ' + @lastName + 
        ' holds the position of ' + @jobTitle

  IF( @jobID = 672 )
     SELECT @newSalary = @salary * 2
  ELSE
  BEGIN
     IF(@jobID = 671 AND DATEDIFF(year, @hireDate, GETDATE()) >= 8)
        SELECT @newSalary = @salary * 1.25
     ELSE
        SELECT @newSalary = @salary * 1.05
  END

  UPDATE EMPLOYEES
  SET SALARY = @newSalary
  WHERE CURRENT OF EMPLOYEE_CURSOR

  FETCH NEXT FROM EMPLOYEE_CURSOR
    INTO @empID, @firstName, @lastName, @jobID, @hireDate, @salary

END

CLOSE EMPLOYEE_CURSOR

DEALLOCATE EMPLOYEE_CURSOR

return

error:
  raiserror @errno @errmsg
  rollback transaction

go


-- 
-- Examples showing differences in how Oracle and SQL Server
-- handle dynamic SQL
--


-- 
-- Dynamic SQL in PL/SQL
--

DECLARE
   sql_stmt    VARCHAR2(200);
   dept_id     NUMBER(2) := 50;
   dept_name   VARCHAR2(14) := 'PERSONNEL';
   location    VARCHAR2(13) := 'DALLAS';

BEGIN
   sql_stmt := 'CREATE TABLE sample1 ' ||
                   (id NUMBER, ' || 
                   'comments VARCHAR2(10) ' ||
                   'last_changed_date DATE');

   EXECUTE IMMEDIATE sql_stmt;

   sql_stmt := 'INSERT INTO dept VALUES (:1, :2, :3)';
   EXECUTE IMMEDIATE sql_stmt USING dept_id, dept_name, location;

END;
/

-- 
-- Dynamic SQL in Transact-SQL
--

DECLARE @SQLString NVARCHAR(200)

select @SQLString = N'select EmployeeID, LastName from ' +
                    N'Employees where EmployeeID = @empid'
 
exec sp_executesql  @SQLString, N'@empid int', @empid = 9

